%**************************************************************************
% readXPMPro.m
%**************************************************************************
%
% reads RHK SM4 data file format
% Data format listed in XPMPro 2.0 User's Manual, Document version 1.0.0
%    (Last modified 11/05 /2008) has ERRORS. 
%
% Use cd command to change path before starting readXPMPro.m
% example:
%   cd('C:\Documents and Settings\All Users\Documents\FromShareRHK\STM_05')
% or
%   cd('P:\STM data\NPF10_03')
%


%**************************************************************************
% primary function
%**************************************************************************
function readXPMPro(process,filename,specScale,ptsAve,imageCorrect,imageScale,extraText)
%   process = 's' for spectra, 't' for topography
%   set filename = '' for file popup
%   set Scale = [1 1 0] no scaling & no offset [scaleX scaleY offset]
%   ptsAve = odd number or set ptsAve = 1 for no averaging
%   imageCorrect 'x' x-scan, 'y' y-scan, 'p' fit plane
%   imageScale = 0 for imagesc(Z) or otherwise image(imageScale*Z)
%   extraText = '' for no text appended to filename in spectra plot
%
% example:
%   readXPMPro('t','09_07_01_Topo_0000.SM4',[0.1 1e9 0.3 0.5 11],25,'x',[0 .35],'0T')
disp('hello')
    % open data file
     if isequal(filename,'')
         [filename, pathname] = uigetfile('*.sm4', 'Pick an SM4-file');
     else
         pathname = pwd; % use current path
     end
     
     if isequal(filename,0)
         disp('User selected Cancel')
     else
%filename = '09_06_20_Spec_0000.SM4';
%pathname = 'P:\STM data\STM_08\';
        disp(['User selected ', fullfile(pathname, filename)])

        % extract filename root for output files
        filenameLen = length(filename);
        filenameRoot = filename(1:filenameLen-4); % remove '.sm4'
        filenameR = fullfile(pathname, filename);
        
        if isequal(process,'t')
            processTopography(filenameR,filenameRoot,imageCorrect,imageScale,extraText); 
        elseif isequal(process,'s')
            processSpectra(filenameR,filenameRoot,specScale,ptsAve,extraText); 
        else
            disp('spec or topo?');
        end
        
      end
%**************************************************************************
% END primary function
%**************************************************************************

%**************************************************************************
% process spectra
%**************************************************************************
function processSpectra(filenameR,filenameRoot,specScale,ptsAve,extraText)
    % open, read, and close SM4 file
    fid = fopen(filenameR,'r');
    
    noSpectra = true;
    RHK = readFileHeader(fid);
    pageHeader = readPageHeader(fid,RHK);
    for i = 1:RHK.fileHeader.totalPageCount % search through page headers for IV data
        if (pageHeader(i).pageType==10)||(pageHeader(i).pageType==16)||(pageHeader(i).pageType==38) % 10 = IV spectra, 16 = image interactive spectra
            spectralDriftData = readSpectralDriftData(fid,pageHeader(i)); % made redundant by function extractSpectraLocations   
            spectralData = readSpectralData(fid,pageHeader(1),RHK.pageData(i).offset);

            % process spectral data
            [specCount,specLocation] = extractSpectraLocations(spectralDriftData,pageHeader(i));
            Xaxis = calculateXaxis(pageHeader(i));
            aveSpectralData = calculateAveSpectralData(pageHeader(i),spectralData,specCount);
            diffSpectralData = calculateDiffSpectralData(pageHeader(i),aveSpectralData,specCount,Xaxis);
            maDiffSpectralData = movingAveDiffSpectralData(pageHeader(i),diffSpectralData,specCount,ptsAve);

            % output data
            plotDiffSpectralDataData(pageHeader(i),Xaxis,maDiffSpectralData,specCount,filenameRoot,specScale,ptsAve,specLocation,extraText);
                % export processed data to ASCII files
        % %        printDiffSpectra(pathname, filenameRoot,RHK,pageHeader(1),Xaxis,diffSpectralData,specCount,specLocation,ptsAve);
        %        printDiffSpectra(pathname, filenameRoot,RHK,pageHeader(1),Xaxis,maDiffSpectralData,specCount,specLocation,ptsAve);
        %        printFileInfo(pathname, filenameRoot,RHK,pageHeader(1), spectralDriftData);
            noSpectra = false;
            break;
        end
    end
    if noSpectra
        disp('no spectra');
    end
    fclose(fid);

%**************************************************************************
% END process spectra
%**************************************************************************
%**************************************************************************
% process topography
%**************************************************************************
function processTopography(filenameR,filenameRoot,imageCorrect,imageScale,extraText)
    % open, read, and close SM4 file
    fid = fopen(filenameR,'r');
    RHK = readFileHeader(fid);
    pageHeader = readPageHeader(fid,RHK);
    noTopo = true;
    for i = 1:RHK.fileHeader.totalPageCount % search through page headers for topographic data
        if (pageHeader(i).pageType==1) % 1 = topographic image. Use first one
            imageData = readImage(fid,pageHeader(i),RHK.pageData(i).offset);
            z = plotImageData(imageData,pageHeader(i),filenameRoot,imageCorrect,imageScale,extraText);
            imwrite(z,'test.bmp','bmp');
            noTopo = false;
            break;
        end
    end
    if noTopo
        disp('no topography');
    end
    fclose(fid);
%**************************************************************************
% END process topography
%**************************************************************************

%**************************************************************************
% plotImageData
%**************************************************************************
function zz = plotImageData(imageData,pageHeader,filenameRoot,imageCorrect,imageScale,extraText)
    filename = strrep(filenameRoot, '_','-');
    Xcenter=(pageHeader.Xoffset/1e-9);
    Ycenter=(pageHeader.Yoffset/1e-9);
    xlen=(pageHeader.width*pageHeader.Xscale/1e-9);
    ylen=(pageHeader.height*pageHeader.Yscale/1e-9);
    deltaX = xlen/pageHeader.width;
    deltaY = ylen/pageHeader.height;
    X(1) = Xcenter-xlen/2+deltaX/2;
    Y(1) = Ycenter-ylen/2+deltaY/2;
    X(2) = Xcenter+xlen/2-deltaX/2;
    Y(2) = Ycenter+ylen/2-deltaY/2;
    
    xx = X(2) - 0.1 *xlen;
    yy = Y(1) + 0.1 *ylen;
    if imageCorrect =='p'
        zz = fitPlaneImageData(imageData,pageHeader);
    else
        zz = fitLineImageData(imageData,imageCorrect,pageHeader);
    end

    if (imageScale(1) < 0)||(imageScale(2) > 1)||(imageScale(1)>=imageScale(2))
        imageScale = [0 1];
        disp('0 <= imageScale <= 1');
    end
    minZ=min(min(zz));
    maxZ=max(max(zz));
    diffZ = maxZ-minZ;
    scaling =[minZ+diffZ*imageScale(1) minZ+diffZ*imageScale(2)];
    im_handle = imagesc(X,Y,zz,scaling);
    anno_obj_handle = text(xx,yy,sprintf('%s %s\n%d*%d [nm2] (%d,%d) [nm]',filename,extraText,abs(round(xlen)),abs(round(ylen)),round(Xcenter),round(Ycenter))); 
    set(anno_obj_handle,'EdgeColor','white','FontSize',18,'Color','white');
    colormap bone;
    axis xy;
    axis square;
%    axis off;
%**************************************************************************
% End plotImageData
%**************************************************************************

%**************************************************************************
% fitPlaneImageData
%**************************************************************************
    function zz = fitPlaneImageData(imageData,pageHeader)
    x = ones(pageHeader.width*pageHeader.height,3);
    y = zeros(pageHeader.width*pageHeader.height,1);
    for i = 1:pageHeader.width
        for j = 1:pageHeader.height
            k=(i-1)*pageHeader.height+j;
            y(k)=imageData(j,i);
            x(k,2)=i;
            x(k,3)=j;
        end
    end
    disp('a:');
    a = x\y;
    clear x;
    clear y;
    z = zeros(pageHeader.height,pageHeader.width);
    for i = 1:pageHeader.width
        for j = 1:pageHeader.height
            z(j,i)=imageData(j,i)-(a(1)+a(2)*i+a(3)*j);
        end
    end
    zz=z';

%**************************************************************************
% END fitLineImageData
%**************************************************************************

%**************************************************************************
% fitLineImageData
%**************************************************************************
    function z = fitLineImageData(imageData,imageCorrect,pageHeader)
    z = zeros(pageHeader.height,pageHeader.width);
    x = zeros(1,pageHeader.width);
    for i = 1:pageHeader.width
        x(i)=i;
    end
    zz=imageData';
    if imageCorrect == 'y'
        for j = 1:pageHeader.width
            a = polyfit(x',zz(:,j),1); % least squares fitting to a line
            z(:,j) = zz(:,j) - (a(1)*x'+a(2));
        end
    else % imageCorrect == 'x'
        for j = 1:pageHeader.width
            a = polyfit(x,zz(j,:),1); % least squares fitting to a line
            z(j,:) = zz(j,:) - (a(1)*x+a(2));
        end
    end
%**************************************************************************
% END fitLineImageData
%**************************************************************************

%**************************************************************************
% readImage
%**************************************************************************
function imageData = readImage(fid,pageHeader,dataOffset)
    fseek(fid,dataOffset,'bof');
    pageHeader.width
    pageHeader.height
%    imageData = zeros(pageHeader.width,pageHeader.height);
    [imageData,count] = fread(fid, [pageHeader.width,pageHeader.height], 'int');
    
%**************************************************************************
% End readImage
%**************************************************************************

%**************************************************************************
% extractSpectraLocations
%**************************************************************************
function [specCount,specLocation] = extractSpectraLocations(spectralDriftData,pageHeader)
    %assume at least one spectrum exists
    i=1;
    specCount = i;
    specLocation(i).X=spectralDriftData(1).Xcoord;
    specLocation(i).Y=spectralDriftData(1).Ycoord;
    for i = 2:pageHeader.height
        if specLocation(specCount).X~=spectralDriftData(i).Xcoord || specLocation(specCount).Y~=spectralDriftData(i).Ycoord
            specCount = specCount+1;
            specLocation(specCount).X=spectralDriftData(i).Xcoord;
            specLocation(specCount).Y=spectralDriftData(i).Ycoord;
        end
    end
%**************************************************************************
% END extractSpectraLocations
%**************************************************************************


%**************************************************************************
% printDiffSpectra
%**************************************************************************
function printDiffSpectra(pathname, filenameRoot,RHK,pageHeader,Xaxis,maDiffSpectralData,specCount,specLocation,ptsAve)
    filenameInf = sprintf('%s%c%s(sm%d).%s', pathname, '\',filenameRoot,ptsAve,'txt');
    fidIV = fopen(filenameInf,'w');
    fprintf(fidIV,'Bias');
    for j =1:specCount
        fprintf(fidIV,'\t[%e,%e]',specLocation(j).X,specLocation(j).Y);
    end
    fprintf(fidIV,'\n');
    for i=1:pageHeader.width-1
        fprintf(fidIV,'%f',Xaxis(i));
        for j =1:specCount
            fprintf(fidIV,'\t%e',maDiffSpectralData(i,j));
        end
        fprintf(fidIV,'\n');
    end
    fclose(fidIV);
%**************************************************************************
% END printDiffSpectra
%**************************************************************************


%**************************************************************************
% calculateDiffSpectralData
%**************************************************************************
function diffSpectralData = calculateDiffSpectralData(pageHeader,aveSpectralData,specCount,Xaxis)
    diffSpectralData = zeros(pageHeader.width-1,specCount);
    for j=1:specCount
        for i=1:pageHeader.width-1
            diffSpectralData(i,j)=( aveSpectralData(i+1,j)-aveSpectralData(i,j) )/(Xaxis(i+1)-Xaxis(i));
        end
    end
    aveSpectralData(1,1);
    aveSpectralData(2,1);
    Xaxis(1);
    Xaxis(2);
    diffSpectralData(1,1);
%**************************************************************************
% END calculateDiffSpectralData
%**************************************************************************
%**************************************************************************
% movingAveDiffSpectralData
%**************************************************************************
function maDiffSpectralData = movingAveDiffSpectralData(pageHeader,diffSpectralData,specCount,ptsAve)
    ptsPerSpec = pageHeader.width-1;
    maDiffSpectralData = zeros(ptsPerSpec,specCount);
    rampUpEnd = (ptsAve-1)/2;
    rampDownStart = (ptsPerSpec+1)-(ptsAve-1)/2;
    for j=1:specCount 
        for k=1:rampUpEnd %ramp moving average at beginning and end of spec
            m=k-1;
            totalBeg = 0.0;
            totalEnd = 0.0;
            for l=k-m:k+m
                totalBeg = totalBeg + diffSpectralData(l,j);
                totalEnd = totalEnd + diffSpectralData((ptsPerSpec+1)-l,j);
            end
            maDiffSpectralData(k,j)= totalBeg/(2*k-1);
            maDiffSpectralData((ptsPerSpec+1)-k,j)= totalEnd/(2*k-1);
        end
        for k=(rampUpEnd + 1):(rampDownStart-1)
            m=(ptsAve-1)/2;
            total = 0.0;
            for l=k-m:k+m
                total = total + diffSpectralData(l,j);
            end
            maDiffSpectralData(k,j)= total/ptsAve;
        end
    end
%**************************************************************************
% END movingAveDiffSpectralData
%**************************************************************************

%**************************************************************************
% plotDiffSpectralDataData
%**************************************************************************
function plotDiffSpectralDataData(pageHeader,Xaxis,diffSpectralData,specCount,filenameRoot,specScale,ptsAve,specLocation,extraText)
    maxX = max(Xaxis)*specScale(1);
    if round(maxX/1)>0
        disp('round(maxX/1)>0');
        adjScale = specScale(1);
        adjXlabel= 'Bias [V]';
        adjBiasLabel= '[V]';
        adjBias = pageHeader.bias*specScale(1);
    elseif round(maxX/0.001)>0
        disp('round(maxX/0.001)>0');
        adjScale = specScale(1)/1e-3;
        adjXlabel= 'Bias [mV]';
        adjBiasLabel= '[mV]';
        adjBias = pageHeader.bias*specScale(1)/1e-3;
    else
        disp('round(maxX/1e-6)>0');
        adjScale = specScale(1)/1e-6;
        adjXlabel= sprintf('%s%s%s','Bias [',texlabel('mu'),'V]');
        adjBiasLabel= '[V]';
        adjBias = pageHeader.bias*specScale(1)/1e-6;
    end
    plot(Xaxis(1:pageHeader.width-1,1)*adjScale,diffSpectralData(:,1)*specScale(2));
    h=get(gcf,'CurrentAxes');
    set(h,'FontSize',16);
    axis tight;
    if (specScale(4) == 0)&&(specScale(5) == 0)
        axis 'auto y';
    else
        ylim([specScale(4) specScale(5)]) 
    end
    grid on;
    x=round(specLocation(1).X/1e-9);
    y=round(specLocation(1).Y/1e-9);

    filename = strrep(filenameRoot, '_','-');
    title(sprintf('%s [sm%d] (%d,%d) %d%s %s',filename,ptsAve,x,y,round(adjBias),adjBiasLabel,extraText));
    xlabel(adjXlabel);
    ylabel('Tunneling Current [nA]');
    hold all % hold plot and cycle line colors
    for i = 2:specCount
        offset = (i-1)*specScale(3);
        plot(Xaxis(1:pageHeader.width-1,1)*adjScale,(diffSpectralData(:,i)*specScale(2)+offset));

    end
    hold off
%**************************************************************************
% END plotDiffSpectralDataData
%**************************************************************************

%**************************************************************************
% calculateAveSpectralData
%**************************************************************************
function aveSpectralData = calculateAveSpectralData(pageHeader,spectralData,specCount)
    aveSpectralData = zeros(pageHeader.width,specCount);
    IVperPoint = pageHeader.height/specCount;
    for point =1:specCount
        curve1 = (point-1)*IVperPoint+1;
        curve2 = point*IVperPoint;
        for voltage = 1:pageHeader.width
            for i = curve1:curve2
                aveSpectralData(voltage,point) = aveSpectralData(voltage,point)+ spectralData(voltage,i);
            end
        end
    end
    aveSpectralData(:,:)=pageHeader.Zscale*aveSpectralData(:,:)/IVperPoint;
%**************************************************************************
% END calculateAveSpectralData
%**************************************************************************


%**************************************************************************
% calculateXaxis
%**************************************************************************
    function Xaxis = calculateXaxis(pageHeader)
    Xaxis = zeros(pageHeader.width,1);
    for voltage = 1:pageHeader.width
        Xaxis(voltage) = pageHeader.Xoffset + voltage*pageHeader.Xscale;
    end
%**************************************************************************
% END calculateXaxis
%**************************************************************************


%**************************************************************************
% plotAveSpectralData
%**************************************************************************
function plotAveSpectralData(Xaxis,aveSpectralData,specCount)
    plot(Xaxis(:,1),aveSpectralData(:,1));
    hold all % hold plot and cycle line colors
    for i = 2:specCount
    plot(Xaxis(:,1),aveSpectralData(:,i));
    end
    hold off
%**************************************************************************
% END plotAveSpectralData
%**************************************************************************


%**************************************************************************
%read spectral data
%**************************************************************************
function spectralData = readSpectralData(fid,pageHeader,IVdataOffset)
    fseek(fid,IVdataOffset,'bof');
    pageHeader.width
    pageHeader.height
     spectralData = zeros(pageHeader.width,pageHeader.height);
     [spectralData,count] = fread(fid, [pageHeader.width,pageHeader.height], 'int');
%**************************************************************************
%END read spectral data
%**************************************************************************

%**************************************************************************
%read spectral drift data
%**************************************************************************
function spectralDriftData = readSpectralDriftData(fid,pageHeader)
    fseek(fid,pageHeader.object(5).offset,'bof');
    for i = 1:pageHeader.height
        spectralDriftData(i).time = fread(fid, 1,'float');
        spectralDriftData(i).Xcoord = fread(fid, 1,'float');
        spectralDriftData(i).Ycoord = fread(fid, 1,'float');
        spectralDriftData(i).dX = fread(fid, 1,'float');
        spectralDriftData(i).dY = fread(fid, 1,'float');
        spectralDriftData(i).cumulativedX = fread(fid, 1,'float');
        spectralDriftData(i).cumulativedY = fread(fid, 1,'float');
%        spectralDrift(i)
    end
%**************************************************************************
% END read spectral drift data
%**************************************************************************

%**************************************************************************
% function read page header string data
%**************************************************************************
function readPageHeaderStringData(fid,pageHeader)
     fseek(fid,pageHeader.object(1).offset,'bof');
     for i = 1:pageHeader.stringCount
         strData = fread(fid, fread(fid, 1,'short'),'short'); % do not keep
     end
%      1. label
%      2. systemText
%      3. sessionText
%      4. userText
%      5. path
%      6. date
%      7. time
%      8. Xunits
%      9. Yunits
%      10. Zunits
%      11. Xlabel
%      12. Ylabel
%      13. statusChannelText
%      14. completedLineCount
%      15. overSamplingCount
%      16. slicedVoltage
%      17. PLLproStatus

%**************************************************************************
% END read page header string data
%**************************************************************************


%**************************************************************************
% function read file header
%**************************************************************************
 function RHK = readFileHeader(fid)
    RHK.fileHeader.Size = fread(fid, 1, 'short');
    RHK.fileHeader.version = fread(fid, 18,'short');
    RHK.fileHeader.totalPageCount = fread(fid, 1, 'int');
    RHK.fileHeader.objectListCount = fread(fid, 1, 'int');
    RHK.fileHeader.objectFieldSize = fread(fid, 1, 'int');
%char(RHK.fileHeader.version)
    fseek(fid,8,'cof'); % reserved for future use

    %Read object list
    for i = 1:RHK.fileHeader.objectListCount
        RHK.fileHeader.object(i).ID = fread(fid, 1, 'int');
        RHK.fileHeader.object(i).offset = fread(fid, 1, 'int');
        RHK.fileHeader.object(i).size = fread(fid, 1, 'int');
disp(sprintf('RHK.fileHeader.object(%d).ID = %d',i,RHK.fileHeader.object(i).ID));
disp(sprintf('RHK.fileHeader.object(%d).offset = %d',i,RHK.fileHeader.object(i).offset));
disp(sprintf('RHK.fileHeader.object(%d).size = %d',i,RHK.fileHeader.object(i).size));
    end
%     objects
%     1. Page Index Header
%     2. PRM Data
%     3. PRM Header

    %read page index header
%    disp('read page index header');
    RHK.pageIndexheader.pageCount = fread(fid, 1, 'int');
    RHK.pageIndexheader.objectListCount = fread(fid, 1, 'int');
%    RHK.pageIndexheader    
    fseek(fid,8,'cof'); % reserved for future use
    
    %read object list
    for i = 1:RHK.pageIndexheader.objectListCount
        RHK.pageIndexheader.object(i).ID = fread(fid, 1, 'int');
        RHK.pageIndexheader.object(i).offset = fread(fid, 1, 'int');
        RHK.pageIndexheader.object(i).size = fread(fid, 1, 'int');
disp(sprintf('RHK.pageIndexheader.object(%d).ID = %d',i,RHK.pageIndexheader.object(i).ID));
disp(sprintf('RHK.pageIndexheader.object(%d).offset = %d',i,RHK.pageIndexheader.object(i).offset));
disp(sprintf('RHK.pageIndexheader.object(%d).size = %d',i,RHK.pageIndexheader.object(i).size));
    end
%     objects
%     1. Page Index Array

    %read page index array
    for i = 1:RHK.fileHeader.totalPageCount % one per page
disp(sprintf('read page index array [%d]',i));
        RHK.pageIndexArray(i).Data1 = fread(fid, 1, 'long');
        RHK.pageIndexArray(i).Data2 = fread(fid, 1, 'short');
        RHK.pageIndexArray(i).Data3 = fread(fid, 1, 'short');
        RHK.pageIndexArray(i).Data4 = fread(fid, 8, 'char');
        RHK.pageIndexArray(i).pageDataType = fread(fid, 1, 'int');
        RHK.pageIndexArray(i).pageSourceType = fread(fid, 1, 'int');
        RHK.pageIndexArray(i).objectListCount = fread(fid, 1, 'int');
        RHK.pageIndexArray(i).minorVersion = fread(fid, 1, 'int');
%RHK.pageIndexArray(i)

        RHK.pageHeader(i).ID = fread(fid, 1, 'int');
        RHK.pageHeader(i).offset = fread(fid, 1, 'int');
        RHK.pageHeader(i).size = fread(fid, 1, 'int');
disp(sprintf('RHK.pageHeader(%d).ID = %d',i,RHK.pageHeader(i).ID));
disp(sprintf('RHK.pageHeader(%d).offset = %d',i,RHK.pageHeader(i).offset));
disp(sprintf('RHK.pageHeader(%d).size = %d',i,RHK.pageHeader(i).size));

        RHK.pageData(i).ID = fread(fid, 1, 'int');
        RHK.pageData(i).offset = fread(fid, 1, 'int');
        RHK.pageData(i).size = fread(fid, 1, 'int');
disp(sprintf('RHK.pageData(%d).ID = %d',i,RHK.pageData(i).ID));
disp(sprintf('RHK.pageData(%d).offset = %d',i,RHK.pageData(i).offset));
disp(sprintf('RHK.pageData(%d).size = %d',i,RHK.pageData(i).size));

        RHK.Thumbnail(i).ID = fread(fid, 1, 'int');
        RHK.Thumbnail(i).offset = fread(fid, 1, 'int');
        RHK.Thumbnail(i).size = fread(fid, 1, 'int');
%RHK.Thumbnail(i)

        RHK.ThumbnailHeader(i).ID = fread(fid, 1, 'int');
        RHK.ThumbnailHeader(i).offset = fread(fid, 1, 'int');
        RHK.ThumbnailHeader(i).size = fread(fid, 1, 'int');
%RHK.ThumbnailHeader(i)
    end
    %RHK
%**************************************************************************
% END function read file header
%**************************************************************************


%**************************************************************************
% function read page header
%**************************************************************************
 function pageHeader = readPageHeader(fid,RHK)
     for j = 1:RHK.fileHeader.totalPageCount
         fseek(fid,RHK.pageHeader(j).offset,'bof');
         pageHeader(j).fieldSize = fread(fid, 1, 'short');
    %     pageHeader(j).signature = fread(fid, 18,'char');
         pageHeader(j).stringCount = fread(fid, 1,'short');
    %     pageHeader(j).type = fread(fid, 1,'int');
         pageHeader(j).pageType = fread(fid, 1,'int');
         pageHeader(j).dataSubSource = fread(fid, 1,'int');
         pageHeader(j).lineType = fread(fid, 1,'int');
         pageHeader(j).x = fread(fid, 1,'int');
         pageHeader(j).y = fread(fid, 1,'int');
         pageHeader(j).width = fread(fid, 1,'int');
         pageHeader(j).height = fread(fid, 1,'int');
    %     pageHeader(j).sourceType = fread(fid, 1,'int');
         pageHeader(j).imageType = fread(fid, 1,'int');
         pageHeader(j).scan = fread(fid, 1,'int');
         pageHeader(j).groupID = fread(fid, 1,'int');
         pageHeader(j).pageDataSize = fread(fid, 1,'uint32');
         pageHeader(j).minZ = fread(fid, 1,'uint32');
         pageHeader(j).maxZ = fread(fid, 1,'uint32');
         pageHeader(j).Xscale = fread(fid, 1,'float');
         pageHeader(j).Yscale = fread(fid, 1,'float');
         pageHeader(j).Zscale = fread(fid, 1,'float');
         pageHeader(j).XYscale = fread(fid, 1,'float');
         pageHeader(j).Xoffset = fread(fid, 1,'float');
         pageHeader(j).Yoffset = fread(fid, 1,'float');
         pageHeader(j).Zoffset = fread(fid, 1,'float');
         pageHeader(j).period = fread(fid, 1,'float');
         pageHeader(j).bias = fread(fid, 1,'float');
         pageHeader(j).current = fread(fid, 1,'float');
         pageHeader(j).angle = fread(fid, 1,'float');
         pageHeader(j).colorInfoListCount = fread(fid, 1,'int');
         pageHeader(j).Xsize = fread(fid, 1,'int');
         pageHeader(j).Ysize = fread(fid, 1,'int');
         pageHeader(j).objectListCount = fread(fid, 1,'int');
         fseek(fid,64,'cof'); % reserved for future use
         for i = 1:pageHeader(j).objectListCount
             pageHeader(j).object(i).ID = fread(fid, 1, 'int');
             pageHeader(j).object(i).offset = fread(fid, 1, 'int');
             pageHeader(j).object(i).size = fread(fid, 1, 'int');
         end
disp(sprintf('pageHeader(%d)',j));
pageHeader(j)
     end
%      objects:     
%      1. String Data
%      2. Image Drift header
%      3. Image Drift Data
%      4. Spec Drift Header
%      5. Spectral Data
%      6. Color Info
%      7. Tip-track Info Header
%      8. Tip-track Data


%**************************************************************************
% end page_header
%**************************************************************************


%**************************************************************************
% print file info
%**************************************************************************
function printFileInfo(pathname, filenameRoot,RHK,pageHeader, spectralDrift)
    filenameInf = sprintf('%s%s.%s', pathname, filenameRoot,'doc');
    fidInf = fopen(filenameInf,'w');
    fprintf(fidInf,'RHK.fileHeader.Size = %d\n',RHK.fileHeader.Size);
    fprintf(fidInf,'RHK.fileHeader.version = ');
    fprintf(fidInf,'%c',RHK.fileHeader.version);
    fprintf(fidInf,'\n');
%    fprintf(fidInf,'RHK. = %d\n',RHK.);
    fprintf(fidInf,'RHK.fileHeader.totalPageCount = %d\n',RHK.fileHeader.totalPageCount);
    fprintf(fidInf,'RHK.fileHeader.objectListCount = %d\n',RHK.fileHeader.objectListCount);
     for i = 1:RHK.fileHeader.objectListCount
        fprintf(fidInf,'RHK.fileHeader.object(%d).ID = %d\n',i,RHK.fileHeader.object(i).ID);
        fprintf(fidInf,'RHK.fileHeader.object(%d).offset = %d\n',i,RHK.fileHeader.object(i).offset);
        fprintf(fidInf,'RHK.fileHeader.object(%d).size = %d\n',i,RHK.fileHeader.object(i).size);
     end
    fprintf(fidInf,'RHK.pageIndexheader.pageCount = %d\n',RHK.pageIndexheader.pageCount);
    fprintf(fidInf,'RHK.pageIndexheader.objectListCount = %d\n',RHK.pageIndexheader.objectListCount);
%     
%     %read object list
     for i = 1:RHK.pageIndexheader.objectListCount
          fprintf(fidInf,'RHK.pageIndexheader.object(%d).ID = %d\n',i,RHK.pageIndexheader.object(i).ID);
%         RHK.objectList(i).ID = fread(fid, 1, 'int');
          fprintf(fidInf,'RHK.pageIndexheader.object(%d).offset = %d\n',i,RHK.pageIndexheader.object(i).offset);
%         RHK.objectList(i).offset = fread(fid, 1, 'int');
          fprintf(fidInf,'RHK.pageIndexheader.object(%d).size = %d\n',i,RHK.pageIndexheader.object(i).size);
%         RHK.objectList(i).size = fread(fid, 1, 'int');
     end
% 
%     %read page index array
     for i = 1:RHK.fileHeader.totalPageCount
% %        sprintf('read page index array [%d]',i)
%         RHK.pageIndexArray(i).Data1 = fread(fid, 1, 'long');
    fprintf(fidInf,'RHK.pageIndexArray(%d).Data1 = %d\n',i,RHK.pageIndexArray(i).Data1);
%         RHK.pageIndexArray(i).Data2 = fread(fid, 1, 'short');
    fprintf(fidInf,'RHK.pageIndexArray(%d).Data2 = %d\n',i,RHK.pageIndexArray(i).Data2);
%         RHK.pageIndexArray(i).Data3 = fread(fid, 1, 'short');
    fprintf(fidInf,'RHK.pageIndexArray(%d).Data3 = %d\n',i,RHK.pageIndexArray(i).Data3);
%         RHK.pageIndexArray(i).Data4 = fread(fid, 8, 'char');
    fprintf(fidInf,'RHK.pageIndexArray(%d).Data4 = ',i);
    fprintf(fidInf,'%c',RHK.pageIndexArray(i).Data4);
    fprintf(fidInf,'\n');
%         RHK.pageIndexArray(i).pageDataType = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageIndexArray(%d).pageDataType = %d\n',i,RHK.pageIndexArray(i).pageDataType);
%         RHK.pageIndexArray(i).pageSourceType = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageIndexArray(%d).pageSourceType = %d\n',i,RHK.pageIndexArray(i).pageSourceType);
%         RHK.pageIndexArray(i).objectListCount = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageIndexArray(%d).objectListCount = %d\n',i,RHK.pageIndexArray(i).objectListCount);
%         RHK.pageIndexArray(i).minorVersion = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageIndexArray(%d).minorVersion = %d\n',i,RHK.pageIndexArray(i).minorVersion);
% %        RHK.pageIndexArray(i)
% 
%         RHK.pageHeader(i).ID = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageHeader(%d).ID = %d\n',i,RHK.pageHeader(i).ID);
%         RHK.pageHeader(i).offset = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageHeader(%d).offset = %d\n',i,RHK.pageHeader(i).offset);
%         RHK.pageHeader(i).size = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageHeader(%d).size = %d\n',i,RHK.pageHeader(i).size);
% %       RHK.pageHeader(i)
% 
%         RHK.pageData(i).ID = fread(fid, 1, 'int');
    fprintf(fidInf,'RHK.pageData(%d).ID = %d\n',i,RHK.pageData(i).ID);
    fprintf(fidInf,'RHK.pageData(%d).offset = %d\n',i,RHK.pageData(i).offset);
    fprintf(fidInf,'RHK.pageData(%d).size = %d\n',i,RHK.pageData(i).size);
%         RHK.pageData(i).offset = fread(fid, 1, 'int');
%         RHK.pageData(i).size = fread(fid, 1, 'int');
% %       RHK.pageData(i)
% 
    fprintf(fidInf,'RHK.Thumbnail(%d).ID = %d\n',i,RHK.Thumbnail(i).ID);
    fprintf(fidInf,'RHK.Thumbnail(%d).offset = %d\n',i,RHK.Thumbnail(i).offset);
    fprintf(fidInf,'RHK.Thumbnail(%d).size = %d\n',i,RHK.Thumbnail(i).size);
%         RHK.Thumbnail(i).ID = fread(fid, 1, 'int');
%         RHK.Thumbnail(i).offset = fread(fid, 1, 'int');
%         RHK.Thumbnail(i).size = fread(fid, 1, 'int');
% %      RHK.Thumbnail(i)
% 
    fprintf(fidInf,'RHK.ThumbnailHeader(%d).ID = %d\n',i,RHK.ThumbnailHeader(i).ID);
    fprintf(fidInf,'RHK.ThumbnailHeader(%d).offset = %d\n',i,RHK.ThumbnailHeader(i).offset);
    fprintf(fidInf,'RHK.ThumbnailHeader(%d).size = %d\n',i,RHK.ThumbnailHeader(i).size);
%         RHK.ThumbnailHeader(i).ID = fread(fid, 1, 'int');
%         RHK.ThumbnailHeader(i).offset = fread(fid, 1, 'int');
%         RHK.ThumbnailHeader(i).size = fread(fid, 1, 'int');
% %      RHK.ThumbnailHeader(i)
     end
%         
        
        fclose(fidInf);
%**************************************************************************
% END print file info
%**************************************************************************
